/*
 * ---------------- www.spacesimulator.net --------------
 *   ---- Space simulators and 3d engine tutorials ----
 *
 * Original Author: Damiano Vitulli
 * Porting to OpenGL3.3: Movania Muhammad Mobeen
 * Shaders Functions: Movania Muhammad Mobeen
 *
 * This program is released under the BSD licence
 * By using this program you agree to licence terms on spacesimulator.net copyright page
 *
 */

/*
 * Spaceships credits:
 * fighter1.3ds - created by: Dario Vitulli 
 * fighter2.3ds - taken from http://www.3dcafe.com/asp/meshes.asp
 * fighter3.3ds - created by: Rene Reiter <renereiter@hotmail.com>
 *
 */ 
#include "texturemappedfont.h"  
#include "framework.h"
#include "load_bmp.h"
#include "load_3ds.h"
#include "load_ini.h"
#include "object.h"
#include "camera.h" 
#include "render_ogl.h"

#include <vector>
using namespace std;
vector<GLuint> vboIDs;
vector<GLuint> vaoIDs;

vector<GLuint> vaoAABBIDs;
vector<GLuint> vboAABBIDs;
 
/*
 *
 * VARIABLES DECLARATION
 *
 */

// FPS calculation and time based physics
unsigned int fps_physics=0;
unsigned int fps_rendering=0;
unsigned int set_fps_physics=100;
unsigned int remaining_time=0; // Remaining time for time based physics
unsigned long fps_count_physics=0; // FPS for physics engine
unsigned long fps_count_rendering=0; // FPS for rendering engine
unsigned long last_ticks=0; // Number of ticks since last frame
int msecxdframe=1000/set_fps_physics; // Milliseconds we want for each physics frame

TextureMappedFont* font1;
TextureMappedFont* font2;
GLSLShader shader;
GLSLShader aabb_shader;

GLsizei stride = sizeof(GLfloat)*3;
void OnShutdown() {
	delete font1;
	delete font2;
	for(int i=0;i<obj_qty;i++){
		if(object[i].id_texture!=-1)
			glDeleteTextures(1, &object[i].id_texture);
	}
	glDeleteBuffers(obj_qty*4, &vboIDs[0]);	
	glDeleteVertexArrays(obj_qty, &vaoIDs[0]);
	glDeleteBuffers(obj_qty, &vboAABBIDs[0]);	
	glDeleteVertexArrays(obj_qty, &vaoAABBIDs[0]);
	vboIDs.clear();
	vaoIDs.clear();
}

void InitShaders(void)
{
	shader.LoadFromFile(GL_VERTEX_SHADER, "shaders/shader.vert");
	shader.LoadFromFile(GL_FRAGMENT_SHADER, "shaders/shader.frag");

	aabb_shader.LoadFromFile(GL_VERTEX_SHADER, "shaders/aabb_shader.vert");	
	aabb_shader.LoadFromFile(GL_FRAGMENT_SHADER, "shaders/aabb_shader.frag");

	shader.CreateAndLinkProgram();
	shader.Use();	
		shader.AddAttribute("vVertex");
		shader.AddAttribute("vUV");
		shader.AddAttribute("vNormal");
		shader.AddUniform("N");
		shader.AddUniform("MV");
		shader.AddUniform("MVP");
		shader.AddUniform("lightPos");
		shader.AddUniform("mat_ambient");
		shader.AddUniform("mat_diffuse");
		shader.AddUniform("mat_specular");
		shader.AddUniform("mat_shininess");
		shader.AddUniform("light_ambient");
		shader.AddUniform("light_diffuse");
		shader.AddUniform("light_specular");
		shader.AddUniform("textureMap");
		shader.AddUniform("has_texture");
		glUniform1i(shader("textureMap"),0);
		glUniform4fv(shader("lightPos"),1, light_position);

		glUniform4fv(shader("light_ambient"),1, light_ambient);
		glUniform4fv(shader("light_diffuse"),1, light_diffuse);
		glUniform4fv(shader("light_specular"),1, light_specular);

		glUniform4fv(shader("mat_ambient"),1, mat_ambient);
		glUniform4fv(shader("mat_diffuse"),1, mat_diffuse);
		glUniform4fv(shader("mat_specular"),1, mat_specular);
		glUniform1fv(shader("mat_shininess"),1, mat_shininess);
	shader.UnUse();

	aabb_shader.CreateAndLinkProgram();
	aabb_shader.Use();		
		aabb_shader.AddAttribute("vVertex");
		aabb_shader.AddUniform("MVP");
		aabb_shader.AddUniform("Color");
	aabb_shader.UnUse();
	GL_CHECK_ERRORS
}
void InitVAO() {
	GL_CHECK_ERRORS
	//Create vao and vbo stuff
	glGenVertexArrays(obj_qty, &vaoIDs[0]);
	glGenBuffers (obj_qty*4, &vboIDs[0]);
	//for AABB
	glGenVertexArrays(obj_qty, &vaoAABBIDs[0]);
	glGenBuffers (obj_qty, &vboAABBIDs[0]);

	GL_CHECK_ERRORS
	int count=0;
	for(int i=0;i<obj_qty;i++){ 
		glBindVertexArray(vaoIDs[i]);	 
			glBindBuffer (GL_ARRAY_BUFFER, vboIDs[count]);
			glBufferData (GL_ARRAY_BUFFER, sizeof(GLfloat)*3*object[i].vertices_qty, &object[i].vertex[0], GL_STATIC_DRAW);
			GL_CHECK_ERRORS
			glEnableVertexAttribArray(shader["vVertex"]);
			glVertexAttribPointer (shader["vVertex"], 3, GL_FLOAT, GL_FALSE,stride,0);
			GL_CHECK_ERRORS
			glBindBuffer (GL_ARRAY_BUFFER, vboIDs[count+1]);
			glBufferData (GL_ARRAY_BUFFER, sizeof(GLfloat)*2*object[i].vertices_qty, &object[i].mapcoord[0], GL_STATIC_DRAW);
			GL_CHECK_ERRORS
			glEnableVertexAttribArray(shader["vUV"]);
			glVertexAttribPointer (shader["vUV"], 2, GL_FLOAT, GL_FALSE,sizeof(GLfloat)*2,0);
			GL_CHECK_ERRORS		
			glBindBuffer (GL_ARRAY_BUFFER, vboIDs[count+2]);
			glBufferData (GL_ARRAY_BUFFER, sizeof(GLfloat)*3*object[i].vertices_qty, &object[i].normal[0], GL_STATIC_DRAW);
			GL_CHECK_ERRORS
			glEnableVertexAttribArray(shader["vNormal"]);		
			glVertexAttribPointer (shader["vNormal"], 3, GL_FLOAT, GL_FALSE,stride,0);
			GL_CHECK_ERRORS
			glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, vboIDs[count+3]);
			glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(GLushort)*3*object[i].polygons_qty, &object[i].polygon[0], GL_STATIC_DRAW);
		glBindVertexArray(0);
		count+=4;

		//now attach the AABB vao and dump the aabb vertices
		glBindVertexArray(vaoAABBIDs[i]);
			glBindBuffer (GL_ARRAY_BUFFER, vboAABBIDs[i]);
			glBufferData (GL_ARRAY_BUFFER, sizeof(GLfloat)*3*8, &object[i].aabb, GL_STATIC_DRAW);
			glEnableVertexAttribArray(aabb_shader["vVertex"]);
			glVertexAttribPointer (aabb_shader["vVertex"], 3, GL_FLOAT, GL_FALSE,stride,0);
		glBindVertexArray(0);
	}

	GL_CHECK_ERRORS
}

/*
 * void WorldInit(void)
 *
 * Used to setup our world
 *  
 */
void WorldInit(void)
{
	LoadINI("world.ini");

	//Lookup tables
	MatrGenerateLookupTab();

	//Fonts loading
	font1=new TextureMappedFont("fonts/font1.bmp");
	font2=new TextureMappedFont("fonts/font2.bmp");	

	//Camera initialization
	MatrIdentity_4x4(camera.matrix);
	
	//Object loading
	ObjLoadFromIni("world.ini","object1");
	ObjLoadFromIni("world.ini","object2");
	ObjLoadFromIni("world.ini","object3");

	InitShaders();	
	//allocate space for the vaos and vbos
	vaoIDs.resize(obj_qty);
	vboIDs.resize(obj_qty*4);
	vaoAABBIDs.resize(obj_qty);
	vboAABBIDs.resize(obj_qty);
	InitVAO();

	//Objects loading (old mode)
	/*
	ObjLoad ("fighter1.3ds","skull.bmp",             -10.0, 0.0, -30.0,    900,0,0);
	ObjLoad ("fighter2.3ds",'\0',                      0.0, 0.0, -30.0,    900,0,0);
	ObjLoad ("fighter3.3ds","spaceshiptexture.bmp",   10.0, 0.0, -30.0,    900,0,0);
	*/
}



/*
 * void FrameworkMainLoop()
 *
 * Main loop, where our simulation runs!
 *
 */

void MainLoop(void)
{
	int i,j; // Counters
	unsigned long l_start_time; // Start time for time based physics
	unsigned long l_elapsed_time; // Elapsed time for time based physics
	unsigned int l_frame_time; // Frame time for time based physics
	char l_collision_value; // Collision result

	// Events
    FrameworkEvents(); // Process incoming events

	// Rendering
	fps_count_rendering++; // Increase rendering FPS counter
	l_start_time = Framework_GetTicks();
	RenderDisplay(); // Draw the screen

	// Elapsed time calculation
	l_elapsed_time=Framework_GetTicks()-l_start_time+remaining_time; // Elapsed time (we add also the previous remaining time)
	l_frame_time=l_elapsed_time / msecxdframe; // Frames quantity we must cycle for physics
	remaining_time=l_elapsed_time % msecxdframe; // Get the remaining time because we are working with integer values

	// Physics
	while (l_frame_time-->0) // Now do physics as many times as we need to match the elapsed time of the rendering phase
	{
		fps_count_physics++; // Increase physics FPS counter

		for(i=0;i<obj_qty;i++) //
		{
			ObjDrag(&object[i]); // Add Drag
			ObjDynamics(&object[i],(float)1.0/(float)set_fps_physics); // Do dynamics
		}

		// Check for collisions: we check only the unique pair
		for(i=0;i<obj_qty-1;i++)
		{
			for(j=i+1;j<obj_qty;j++)
			{
				l_collision_value=ObjCollisionCheck(&object[i],&object[j]); // Check all the unique objects pairs
				if (l_collision_value==2) // If the objects are colliding
				{
					ObjCollisionResponse(&object[i],&object[j]); // Collision response
					FrameworkAudioPlayWave("sounds/impa1019.wav"); // Play a CRASH!
				}
			} //j
		} //i
	}//while (l_frame_time-->0)
	

	// Rendering and Physics FPS calculation
	if ((Framework_GetTicks()-last_ticks)>=1000) // Every second
	{
		last_ticks = Framework_GetTicks(); // Save the current ticks to catch the next second
		// Assings the current FPS count to the global variables
		fps_physics=fps_count_physics;
		fps_rendering=fps_count_rendering;
		// Clear the local counters
		fps_count_physics = 0;
		fps_count_rendering = 0;
	}
}



/*
 * void KeyboardHandle(int p_unicode)
 *
 * This function handles keyboard events
 *
 */

void KeyboardHandle(int p_unicode)
{
	switch (p_unicode)
	{
	case 106: // j
		ObjForce(&object[obj_control],1,0,0,-50000);
		break;
	case 109: // m
		ObjForce(&object[obj_control],1,0,0,50000);
		break;
	case 107: // k
		ObjTorque(&object[obj_control],0,50000,0);
		break;
	case 108: // l
		ObjTorque(&object[obj_control],0,-50000,0);
		break;
	case 101: // e
		CamRotate(200,0,0);
		break;
	case 99: // c
		CamRotate(-200,0,0);
		break;
	case 97: // a
		CamRotate(0,-200,0);
		break;
	case 100: // d
		CamRotate(0,200,0);
		break;
	case 122: // z
		CamRotate(0,0,-200);
		break;
	case 120: // x
		CamRotate(0,0,200);
		break;
	case 119: // w
		CamTranslate(0,0,1);
		break;
	case 115: // s
		CamTranslate(0,0,-1);
		break;
	case 114: // r
		render_filling++;
		if (render_filling>1) render_filling=0;
		break;
	case 273: // UP ARROW
		ObjTorque(&object[obj_control],-50000,0,0);
		break;
	case 274: // DOWN_ARROW
		ObjTorque(&object[obj_control],50000,0,0);
		break;
	case 276: // LEFT ARROW
		ObjTorque(&object[obj_control],0,0,50000);
		break;
	case 275: // RIGHT ARROW
		ObjTorque(&object[obj_control],0,0,-50000);
		break;
	case 280: // PAGEUP
		obj_control++;
		camera.n_object++;
		if (obj_control>=obj_qty) obj_control=0;
		if (camera.n_object>=obj_qty) camera.n_object=0;
		break;
	case 281: // PAGEDOWN
		obj_control--;
		camera.n_object--;
		if (obj_control<0) obj_control=obj_qty-1;
		if (camera.n_object<0) camera.n_object=obj_qty-1;
		break;
	case 27:// ESCAPE
		FrameworkQuit();
		break;
	default:                                    
		break;                                  
	}
}



/*
 * The main routine
 *  
 */ 

#include <iostream>
using namespace std;

int main(int argc, char** argv) {
	atexit(OnShutdown);
	// Standard output as log file
	freopen( "log.txt", "w", stdout );
	freopen( "log.txt", "w", stderr );

	// Initializations
	FrameworkInit(&argc,argv); // Framework		
	RenderInit(); // Graphic library
	glewExperimental = GL_TRUE;
	GLenum err = glewInit();
	if (GLEW_OK != err)	{
		cerr<<"Error: "<<glewGetErrorString(err)<<endl;
	} else {
		if (GLEW_VERSION_3_3)
		{
			cout<<"Driver supports OpenGL 3.3\nDetails:"<<endl;
		}
	}
	cout<<"Using GLEW "<<glewGetString(GLEW_VERSION)<<endl;
	cout<<"Vendor: "<<glGetString (GL_VENDOR)<<endl;
	cout<<"Renderer: "<<glGetString (GL_RENDERER)<<endl;
	cout<<"Version: "<<glGetString (GL_VERSION)<<endl;
	cout<<"GLSL: "<<glGetString (GL_SHADING_LANGUAGE_VERSION)<<endl;

    // Set the GLUT function
	WorldInit(); // World

	// Main loop
    FrameworkMainLoop();

    return 0;
}