/*
 * ---------------- www.spacesimulator.net --------------
 *   ---- Space simulators and 3d engine tutorials ----
 *
 * Author: Damiano Vitulli
 *
 * This program is released under the BSD licence
 * By using this program you agree to licence terms on spacesimulator.net copyright page
 *
 *
 * Framework functions (GLUT)
 *  
 */
#ifdef FRAMEWORK_GLUT
#include <GL/glew.h>
#include "framework.h"
#include "object.h"
#include "camera.h"
#include "render_ogl.h"
#include "malloc.h"
#include "load_ini.h"
#include "main.h"
#include <iostream>


using namespace std;

/*
 *
 * VARIABLES DECLARATION
 *
 */

// Main window configuration
int framework_screen_width=640; // Width
int framework_screen_height=480; // Height
int framework_screen_depth; // Color depth
int framework_fullscreen=0; // Fullscreen
int hW = framework_screen_width/2;
int hH = framework_screen_height/2;
void keyboard(unsigned char p_key, int p_x, int p_y);
void keyboard_s (int p_key, int p_x, int py);

#ifdef USE_GLM
#include <glm/glm.hpp>
#include <glm/gtc/matrix_projection.hpp>
#include <glm/gtc/matrix_transform.hpp>
#include <glm/gtc/type_ptr.hpp>
#else
#include "mat_matr.h"
#endif

extern 
#ifdef USE_GLM
glm::mat4 
#else 	
matrix_4x4_type 
#endif
P;//projection matrix;



/*
 * int FrameworkInit()
 *
 * This function initializes the framework (rendering device, input and sound system)
 *
 * Return value: (char) 1 if the framework was initialized correctly otherwise it exit with ERRORLEVEL 0
 *
 */

int FrameworkInit(int *argcp,char *argv[])
{ 

    // We use the GLUT utility to initialize the window, to handle the input and to interact with the windows system
    fprintf( stdout, "GLUT Framework initialization\n");
    glutInit(argcp,argv);    

    // Now we load the video settings from config.ini
	LoadINI("config.ini");
	framework_screen_width=LoadINI_GetInt("config.ini","screen settings","width",640);
	framework_screen_height=LoadINI_GetInt("config.ini","screen settings","height",480);
	hW = framework_screen_width/2;
	hH = framework_screen_height/2;
    glutInitDisplayMode(GLUT_DEPTH | GLUT_DOUBLE | GLUT_RGB);	
	glutInitContextVersion (3, 3);
	glutInitContextFlags (GLUT_CORE_PROFILE | GLUT_DEBUG);
	glutInitWindowSize(framework_screen_width, framework_screen_height);
	#ifdef USE_GLM
	glutCreateWindow("Tutorial 9 - OpenGL 3.3 [Using GLM matrix library]");
	#else
	glutCreateWindow("Tutorial 9 - OpenGL 3.3 [Using spacesimulator.net matrix library.]");
	#endif

	glewExperimental = GL_TRUE;
	GLenum err = glewInit();
	if (GLEW_OK != err)	{
		cerr<<"Error: "<<glewGetErrorString(err)<<endl;
	} else {
		if (GLEW_VERSION_3_3)
		{
			cout<<"Driver supports OpenGL 3.3\nDetails:"<<endl;
		}
	}
	cout<<"Using GLEW "<<glewGetString(GLEW_VERSION)<<endl;
	cout<<"Vendor: "<<glGetString (GL_VENDOR)<<endl;
	cout<<"Renderer: "<<glGetString (GL_RENDERER)<<endl;
	cout<<"Version: "<<glGetString (GL_VERSION)<<endl;
	cout<<"GLSL: "<<glGetString (GL_SHADING_LANGUAGE_VERSION)<<endl;

    // Set the GLUT function
	cout<<"Setting GLUT functions"<<endl;
    glutDisplayFunc(MainLoop);
    glutIdleFunc(MainLoop);
    glutReshapeFunc (FrameworkResize);
    glutKeyboardFunc (keyboard);
    glutSpecialFunc (keyboard_s);

    cout<<"Framework initialized correctly"<<endl;
	return (1);
}



/*
 * void FrameworkQuit()
 *
 * Used to exit from the program
 *
 */

void FrameworkQuit()
{
    fprintf( stdout, "Exiting program\n");
	exit(0);
}



/*
 * void FrameworkMainLoop()
 *
 * Main loop, where our simulation runs!
 *
 */

void FrameworkMainLoop()
{
	fprintf( stdout, "Entering main loop\n");
	glutMainLoop();
}



/*
 * void FrameworkResize(int p_width, int p_height)
 *
 * This routine must be called everytime we resize our window.
 *
 * Input parameters: p_width = width in pixels of our viewport
 *					 p_height = height in pixels of our viewport
 * 
 */

void FrameworkResize (int p_width, int p_height)
{
	if (framework_screen_width==0 && framework_screen_height==0) exit(0);
    framework_screen_width=p_width; // We obtain the new screen width values and store it
    framework_screen_height=p_height; // Height value
    hW = framework_screen_width/2;
	hH = framework_screen_height/2;
	
	glViewport (0, 0, (GLsizei) framework_screen_width, (GLsizei) framework_screen_height);

	//setup the projection matrix
	#ifdef USE_GLM
		P = glm::perspective(45.0f, (GLfloat)framework_screen_width/framework_screen_height, 5.0f, 10000.f);
	#else
		Perspective(45.0f, (GLfloat)framework_screen_width/framework_screen_height, 5.0f, 10000.f, P);
	#endif
       
    glutPostRedisplay (); // This command redraw the scene (it calls the same routine of glutDisplayFunc)
}



/*
 * void FrameworkEvents()
 *
 * Subroutine used to handle some events (keyboard, joystick inputs, framework quit)
 *
 */

void FrameworkEvents()
{
	// Glut handles events using functions callbacks
}



/*
 * long Framework_GetTicks()
 *
 * Used to get the time in msec for time based animation and FPS calculation
 *
 */

long Framework_GetTicks()
{
	//In GLUT we have not functions to get system ticks so we return always +10
	//WARNING Physics is NOT time corrected!
	static long s_ticks;  
	return (s_ticks=s_ticks+10);
}



/*
 * void FrameworkSwapBuffers()
 *
 * Subroutine used to swap the color-buffers
 *
 * In double buffered mode the visible buffer is replaced with the writing buffer
 *
 */

void FrameworkSwapBuffers() 
{
	glutSwapBuffers( );
}



/*
 * void keyboard(unsigned char p_key, int p_x, int p_y)
 *
 * Used to handle the keyboard input (ASCII Characters)
 *
 * Input parameters: p_key = ascii code of the key pressed
 *					 p_x = not used, to maintain compatibility with glut function
 *                   p_y = not used, to maintain compatibility with glut function
 * 
 */

void keyboard(unsigned char p_key, int p_x, int p_y)
{  
	KeyboardHandle((int)p_key);
}



/*
 * void keyboard(int p_key, int p_x, int py)
 *
 * Used to handle the keyboard input (not ASCII Characters)
 *
 * Input parameters: p_key = code of the key pressed
 *					 p_x = not used, to maintain compatibility with glut function
 *                   p_y = not used, to maintain compatibility with glut function
 * 
 */

void keyboard_s (int p_key, int p_x, int py)
{
    switch (p_key)
    {
        case GLUT_KEY_UP:
            KeyboardHandle(273);
        break;
        case GLUT_KEY_DOWN:
            KeyboardHandle(274);
        break;
        case GLUT_KEY_LEFT:
            KeyboardHandle(276);
        break;
        case GLUT_KEY_RIGHT:
            KeyboardHandle(275);
        break;
		case GLUT_KEY_PAGE_UP:
            KeyboardHandle(280);
        break;
        case GLUT_KEY_PAGE_DOWN:
            KeyboardHandle(281);
        break;

    }
}



/*
 * void FrameworkAudioPlayWave(char *p_filename)
 *
 * To mix and play sounds
 *
 * Input parameters: p_filename = Filename of the sound to play
 *
 */

void FrameworkAudioPlayWave(char *file)
{
	// Glut has not audio
}

#endif